<?php

namespace App\Service;

use App\Entity\Fournisseur;
use App\Entity\Achat;
use App\Repository\AchatRepository;
use App\Repository\PaiementRepository;

class FournisseurService
{
    public function __construct(
        private AchatRepository $achatRepository,
        private PaiementRepository $paiementRepository
    ) {
    }

    /**
     * Calcule le total des achats pour un fournisseur
     */
    public function getTotalAchats(Fournisseur $fournisseur): float
    {
        return $fournisseur->getTotalAchete();
    }

    /**
     * Calcule le total payé pour un fournisseur
     */
    public function getTotalPaye(Fournisseur $fournisseur): float
    {
        return $this->paiementRepository->getTotalPayeByFournisseur($fournisseur->getId());
    }

    /**
     * Calcule le reste dû pour un fournisseur
     */
    public function getResteDu(Fournisseur $fournisseur): float
    {
        return $this->getTotalAchats($fournisseur) - $this->getTotalPaye($fournisseur);
    }

    /**
     * Récupère le dernier achat
     */
    public function getDernierAchat(Fournisseur $fournisseur): ?Achat
    {
        $achats = $this->achatRepository->findBy(
            ['fournisseur' => $fournisseur],
            ['dateAchat' => 'DESC'],
            1
        );

        return !empty($achats) ? $achats[0] : null;
    }

    /**
     * Récupère le prochain paiement prévu
     */
    public function getProchainPaiement(Fournisseur $fournisseur): ?Achat
    {
        $achats = $this->achatRepository->createQueryBuilder('a')
            ->where('a.fournisseur = :fournisseur')
            ->andWhere('a.dateLimitePaiement IS NOT NULL')
            ->andWhere('a.dateLimitePaiement >= :aujourdhui')
            ->setParameter('fournisseur', $fournisseur)
            ->setParameter('aujourdhui', new \DateTime())
            ->orderBy('a.dateLimitePaiement', 'ASC')
            ->setMaxResults(1)
            ->getQuery()
            ->getResult();

        return !empty($achats) ? $achats[0] : null;
    }

    /**
     * Compte les paiements en retard
     */
    public function getPaiementsEnRetard(Fournisseur $fournisseur): array
    {
        return $this->achatRepository->createQueryBuilder('a')
            ->where('a.fournisseur = :fournisseur')
            ->andWhere('a.dateLimitePaiement IS NOT NULL')
            ->andWhere('a.dateLimitePaiement < :aujourdhui')
            ->andWhere('a.montantRestant > 0')
            ->setParameter('fournisseur', $fournisseur)
            ->setParameter('aujourdhui', new \DateTime())
            ->orderBy('a.dateLimitePaiement', 'ASC')
            ->getQuery()
            ->getResult();
    }

    /**
     * Récupère les articles les plus achetés
     */
    public function getArticlesPlusAchetes(Fournisseur $fournisseur, int $limit = 5): array
    {
        $qb = $this->achatRepository->createQueryBuilder('a')
            ->join('a.lignesAchat', 'l')
            ->join('l.produit', 'p')
            ->where('a.fournisseur = :fournisseur')
            ->setParameter('fournisseur', $fournisseur)
            ->select('p.nom as produit_nom, SUM(l.quantite) as total_quantite')
            ->groupBy('p.id', 'p.nom')
            ->orderBy('total_quantite', 'DESC')
            ->setMaxResults($limit);

        return $qb->getQuery()->getResult();
    }

    /**
     * Calcule le nombre total d'articles achetés
     */
    public function getTotalArticlesAchetes(Fournisseur $fournisseur): int
    {
        $result = $this->achatRepository->createQueryBuilder('a')
            ->join('a.lignesAchat', 'l')
            ->where('a.fournisseur = :fournisseur')
            ->setParameter('fournisseur', $fournisseur)
            ->select('SUM(l.quantite) as total')
            ->getQuery()
            ->getSingleScalarResult();

        return (int)($result ?? 0);
    }
}

