<?php

namespace App\Entity;

use App\Repository\AchatRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Entity(repositoryClass: AchatRepository::class)]
#[ORM\Table(name: 'achats')]
class Achat
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 50, unique: true)]
    private ?string $numeroFacture = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $referenceFacture = null; // Référence facture optionnelle

    #[ORM\ManyToOne(inversedBy: 'achats')]
    #[ORM\JoinColumn(nullable: false)]
    private ?Fournisseur $fournisseur = null;

    #[ORM\ManyToOne(inversedBy: 'achats')]
    #[ORM\JoinColumn(nullable: false)]
    private ?Magasin $magasin = null;

    #[ORM\ManyToOne]
    #[ORM\JoinColumn(nullable: false)]
    private ?User $utilisateur = null;

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2)]
    private ?string $montantTotal = null;

    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    private ?\DateTimeInterface $dateAchat = null;

    #[ORM\OneToMany(mappedBy: 'achat', targetEntity: LigneAchat::class, cascade: ['persist', 'remove'])]
    private Collection $lignesAchat;

    #[ORM\OneToMany(mappedBy: 'achat', targetEntity: Paiement::class, cascade: ['persist', 'remove'])]
    private Collection $paiements;

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2)]
    private ?string $montantRestant = null; // Montant restant à payer

    #[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
    private ?\DateTimeInterface $dateLimitePaiement = null; // Date limite de paiement

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    private ?string $notes = null;

    public function __construct()
    {
        $this->lignesAchat = new ArrayCollection();
        $this->paiements = new ArrayCollection();
        $this->dateAchat = new \DateTime();
        $this->montantRestant = '0.00';
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getNumeroFacture(): ?string
    {
        return $this->numeroFacture;
    }

    public function setNumeroFacture(string $numeroFacture): self
    {
        $this->numeroFacture = $numeroFacture;
        return $this;
    }

    public function getFournisseur(): ?Fournisseur
    {
        return $this->fournisseur;
    }

    public function setFournisseur(?Fournisseur $fournisseur): self
    {
        $this->fournisseur = $fournisseur;
        return $this;
    }

    public function getMagasin(): ?Magasin
    {
        return $this->magasin;
    }

    public function setMagasin(?Magasin $magasin): self
    {
        $this->magasin = $magasin;
        return $this;
    }

    public function getUtilisateur(): ?User
    {
        return $this->utilisateur;
    }

    public function setUtilisateur(?User $utilisateur): self
    {
        $this->utilisateur = $utilisateur;
        return $this;
    }

    public function getMontantTotal(): ?string
    {
        return $this->montantTotal;
    }

    public function setMontantTotal(string $montantTotal): self
    {
        $this->montantTotal = $montantTotal;
        return $this;
    }

    public function getDateAchat(): ?\DateTimeInterface
    {
        return $this->dateAchat;
    }

    public function setDateAchat(\DateTimeInterface $dateAchat): self
    {
        $this->dateAchat = $dateAchat;
        return $this;
    }

    /**
     * @return Collection<int, LigneAchat>
     */
    public function getLignesAchat(): Collection
    {
        return $this->lignesAchat;
    }

    public function addLigneAchat(LigneAchat $ligneAchat): self
    {
        if (!$this->lignesAchat->contains($ligneAchat)) {
            $this->lignesAchat->add($ligneAchat);
            $ligneAchat->setAchat($this);
        }
        return $this;
    }

    public function getNotes(): ?string
    {
        return $this->notes;
    }

    public function setNotes(?string $notes): self
    {
        $this->notes = $notes;
        return $this;
    }

    public function getReferenceFacture(): ?string
    {
        return $this->referenceFacture;
    }

    public function setReferenceFacture(?string $referenceFacture): self
    {
        $this->referenceFacture = $referenceFacture;
        return $this;
    }

    public function getMontantRestant(): ?string
    {
        return $this->montantRestant;
    }

    public function setMontantRestant(string $montantRestant): self
    {
        $this->montantRestant = $montantRestant;
        return $this;
    }

    public function getDateLimitePaiement(): ?\DateTimeInterface
    {
        return $this->dateLimitePaiement;
    }

    public function setDateLimitePaiement(?\DateTimeInterface $dateLimitePaiement): self
    {
        $this->dateLimitePaiement = $dateLimitePaiement;
        return $this;
    }

    /**
     * @return Collection<int, Paiement>
     */
    public function getPaiements(): Collection
    {
        return $this->paiements;
    }

    public function addPaiement(Paiement $paiement): self
    {
        if (!$this->paiements->contains($paiement)) {
            $this->paiements->add($paiement);
            $paiement->setAchat($this);
        }
        return $this;
    }

    public function removePaiement(Paiement $paiement): self
    {
        if ($this->paiements->removeElement($paiement)) {
            if ($paiement->getAchat() === $this) {
                $paiement->setAchat(null);
            }
        }
        return $this;
    }

    /**
     * Calcule le montant total payé pour cet achat
     */
    public function getMontantPaye(): float
    {
        $total = 0;
        foreach ($this->paiements as $paiement) {
            $total += (float)$paiement->getMontant();
        }
        return $total;
    }

    /**
     * Détermine l'état du paiement
     */
    public function getEtatPaiement(): string
    {
        $montantPaye = $this->getMontantPaye();
        $montantTotal = (float)$this->montantTotal;

        if ($montantPaye == 0) {
            return 'non_paye';
        } elseif ($montantPaye >= $montantTotal) {
            return 'paye';
        } else {
            return 'partiel';
        }
    }
}

