<?php

namespace App\Controller;

use App\Entity\Vente;
use App\Entity\LigneVente;
use App\Entity\Client;
use App\Entity\PaiementClient;
use App\Entity\Produit;
use App\Repository\VenteRepository;
use App\Repository\ProduitRepository;
use App\Repository\ClientRepository;
use App\Service\LogService;
use App\Service\StockService;
use App\Service\FactureService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/ventes', name: 'ventes_')]
#[IsGranted('ROLE_USER')]
class VenteController extends AbstractController
{
    #[Route('', name: 'index', methods: ['GET'])]
    public function index(VenteRepository $venteRepository, Request $request): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        // Filtres
        $dateDebut = $request->query->get('date_debut') ? new \DateTime($request->query->get('date_debut')) : new \DateTime('first day of this month');
        $dateFin = $request->query->get('date_fin') ? new \DateTime($request->query->get('date_fin')) : new \DateTime('now');
        $clientId = $request->query->get('client') ? (int)$request->query->get('client') : null;
        $statutPaiement = $request->query->get('statut_paiement');
        $produitId = $request->query->get('produit') ? (int)$request->query->get('produit') : null;

        $ventes = $venteRepository->findVentesByMagasinAndDate($magasin, $dateDebut, $dateFin);

        // Appliquer les filtres supplémentaires
        if ($clientId) {
            $ventes = array_filter($ventes, fn($v) => $v->getClient() && $v->getClient()->getId() === $clientId);
        }
        if ($statutPaiement) {
            $ventes = array_filter($ventes, fn($v) => $v->getStatutPaiement() === $statutPaiement);
        }
        if ($produitId) {
            $ventes = array_filter($ventes, function($v) use ($produitId) {
                foreach ($v->getLignesVente() as $ligne) {
                    if ($ligne->getProduit()->getId() === $produitId) {
                        return true;
                    }
                }
                return false;
            });
        }

        $clients = $magasin->getClients()->toArray();

        return $this->render('vente/index.html.twig', [
            'ventes' => array_values($ventes),
            'clients' => $clients,
            'filters' => [
                'date_debut' => $dateDebut->format('Y-m-d'),
                'date_fin' => $dateFin->format('Y-m-d'),
                'client' => $clientId,
                'statut_paiement' => $statutPaiement,
                'produit' => $produitId,
            ],
            'dateDebut' => $dateDebut,
            'dateFin' => $dateFin,
        ]);
    }

    #[Route('/nouvelle', name: 'nouvelle', methods: ['GET', 'POST'])]
    public function nouvelle(
        Request $request,
        EntityManagerInterface $em,
        ProduitRepository $produitRepository,
        ClientRepository $clientRepository,
        LogService $logService,
        StockService $stockService,
        FactureService $factureService
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        // Si c'est une requête AJAX pour ajouter un produit
        if ($request->isXmlHttpRequest() && $request->isMethod('POST')) {
            $codeBarres = $request->request->get('code_barres');
            $produitId = $request->request->get('produit_id');
            
            if ($codeBarres) {
                $produit = $produitRepository->findOneBy(['codeBarres' => $codeBarres, 'magasin' => $magasin, 'actif' => true]);
            } elseif ($produitId) {
                $produit = $produitRepository->find($produitId);
            } else {
                return new JsonResponse(['error' => 'Code-barres ou ID produit requis'], 400);
            }

            if (!$produit || $produit->getMagasin() !== $magasin) {
                return new JsonResponse(['error' => 'Produit non trouvé'], 404);
            }

            // Vérifier que le produit est disponible (pas vendu et avec stock > 0)
            if ($produit->getStatut() === Produit::STATUT_VENDU || $produit->getQuantite() <= 0) {
                return new JsonResponse(['error' => 'Produit non disponible (statut: ' . $produit->getStatut() . ', stock: ' . $produit->getQuantite() . ')'], 400);
            }

            // Vérifier l'IMEI si c'est un téléphone
            if ($produit->getImei()) {
                $venteExistante = $em->getRepository(Vente::class)->createQueryBuilder('v')
                    ->join('v.lignesVente', 'lv')
                    ->where('lv.produit = :produit')
                    ->andWhere('v.retour = false')
                    ->setParameter('produit', $produit)
                    ->getQuery()
                    ->getOneOrNullResult();
                
                if ($venteExistante) {
                    return new JsonResponse(['error' => 'Ce produit (IMEI: ' . $produit->getImei() . ') a déjà été vendu'], 400);
                }
            }

            return new JsonResponse([
                'id' => $produit->getId(),
                'nom' => $produit->getNom(),
                'codeBarres' => $produit->getCodeBarres(),
                'referenceInterne' => $produit->getReferenceInterne(),
                'imei' => $produit->getImei(),
                'numeroSerie' => $produit->getNumeroSerie(),
                'prixAchat' => $produit->getPrixAchat(),
                'prixVenteRecommande' => $produit->getPrixVenteRecommande(),
                'margeMinimale' => $produit->getMargeMinimale(),
                'quantite' => $produit->getQuantite(),
                'categorie' => $produit->getCategorie()->getNom(),
                'sousCategorie' => $produit->getSousCategorie() ? $produit->getSousCategorie()->getNom() : null,
                'caracteristiques' => $produit->getCaracteristiquesValeurs(),
            ]);
        }

        // Récupérer les sous-catégories avec produits disponibles pour l'affichage initial
        $sousCategories = $em->getRepository(\App\Entity\SousCategorie::class)
            ->createQueryBuilder('sc')
            ->join('sc.produits', 'p')
            ->where('sc.magasin = :magasin')
            ->andWhere('p.actif = true')
            ->andWhere('p.quantite > 0')
            ->andWhere('p.statut != :statutVendu')
            ->setParameter('magasin', $magasin)
            ->setParameter('statutVendu', Produit::STATUT_VENDU)
            ->groupBy('sc.id')
            ->orderBy('sc.nom', 'ASC')
            ->getQuery()
            ->getResult();
        
        $clients = $clientRepository->findBy(['magasin' => $magasin], ['nom' => 'ASC']);

        return $this->render('vente/nouvelle.html.twig', [
            'sousCategories' => $sousCategories,
            'clients' => $clients,
        ]);
    }

    #[Route('/finaliser', name: 'finaliser', methods: ['POST'])]
    public function finaliser(
        Request $request,
        EntityManagerInterface $em,
        ProduitRepository $produitRepository,
        ClientRepository $clientRepository,
        LogService $logService,
        StockService $stockService,
        FactureService $factureService
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $data = json_decode($request->getContent(), true);

        if (!$data || !isset($data['lignes']) || empty($data['lignes'])) {
            return new JsonResponse(['error' => 'Aucun produit à vendre'], 400);
        }

        // Créer la vente
        $vente = new Vente();
        $vente->setVendeur($user);
        $vente->setMagasin($magasin);
        $vente->setNumeroFacture($factureService->genererNumeroFacture());
        $vente->setDateVente(new \DateTime());

        // Gérer le client
        $clientId = $data['client_id'] ?? null;
        $clientRapide = $data['client_rapide'] ?? null;
        
        if ($clientId) {
            $client = $clientRepository->find($clientId);
            if ($client && $client->getMagasin() === $magasin) {
                $vente->setClient($client);
            }
        } elseif ($clientRapide && isset($clientRapide['nom'])) {
            // Créer un client rapide (optionnel, peut être null)
            // Pour l'instant, on laisse client = null pour "client rapide"
        }

        // Persister la vente AVANT de créer les mouvements de stock
        // pour éviter l'erreur de cascade persist
        $em->persist($vente);
        // Ne pas flush ici, on flush à la fin une seule fois

        // Traiter les lignes de vente
        $montantTotal = 0;
        foreach ($data['lignes'] as $ligneData) {
            $produit = $produitRepository->find($ligneData['produit_id']);
            if (!$produit || $produit->getMagasin() !== $magasin) {
                continue;
            }

            // Vérifier la disponibilité
            if ($produit->getStatut() !== Produit::STATUT_EN_STOCK) {
                return new JsonResponse(['error' => "Le produit {$produit->getNom()} n'est plus disponible"], 400);
            }

            $quantite = (int)($ligneData['quantite'] ?? 1);
            if ($quantite > $produit->getQuantite()) {
                return new JsonResponse(['error' => "Stock insuffisant pour {$produit->getNom()}"], 400);
            }

            $prixVente = (float)($ligneData['prix_vente'] ?? $produit->getPrixVenteRecommande() ?? $produit->getPrixAchat());
            
            // Vérifier la marge minimale
            $marge = $prixVente - (float)$produit->getPrixAchat();
            if ($produit->getMargeMinimale() && $marge < (float)$produit->getMargeMinimale()) {
                // Alerte mais on continue (sera affichée dans l'interface)
            }

            $ligne = new LigneVente();
            $ligne->setVente($vente);
            $ligne->setProduit($produit);
            $ligne->setQuantite($quantite);
            $ligne->setPrixUnitaire((string)$prixVente);
            $ligne->setPrixAchatUnitaire($produit->getPrixAchat());
            $ligne->setMontantTotal((string)($prixVente * $quantite));
            
            $vente->addLigneVente($ligne);
            $montantTotal += (float)$ligne->getMontantTotal();

            // Mettre à jour le stock
            if ($quantite >= $produit->getQuantite()) {
                $produit->setStatut(Produit::STATUT_VENDU);
                $produit->setQuantite(0);
            } else {
                $produit->setQuantite($produit->getQuantite() - $quantite);
            }

            // Enregistrer le mouvement de stock
            $stockService->enregistrerMouvement(
                $produit,
                \App\Entity\MouvementStock::TYPE_SORTIE_VENTE,
                $quantite,
                $user,
                "Vente #{$vente->getNumeroFacture()}",
                $vente
            );
        }

        // Calculer les montants
        $reduction = (float)($data['reduction'] ?? 0);
        $reductionType = $data['reduction_type'] ?? 'montant'; // 'montant' ou 'pourcentage'
        
        if ($reductionType === 'pourcentage') {
            $reduction = ($montantTotal * $reduction) / 100;
        }

        $vente->setMontantTotal((string)$montantTotal);
        $vente->setReduction((string)$reduction);
        $vente->setMontantFinal((string)($montantTotal - $reduction));

        // Gérer les paiements
        $paiements = $data['paiements'] ?? [];
        $montantPaye = 0;
        
        foreach ($paiements as $paiementData) {
            $paiement = new PaiementClient();
            $paiement->setVente($vente);
            if ($vente->getClient()) {
                $paiement->setClient($vente->getClient());
            }
            $paiement->setUtilisateur($user);
            $paiement->setMontant((string)$paiementData['montant']);
            $paiement->setMode($paiementData['mode']);
            $paiement->setReference($paiementData['reference'] ?? null);
            $paiement->setNotes($paiementData['notes'] ?? null);
            $paiement->setDatePaiement(new \DateTime($paiementData['date'] ?? 'now'));
            
            $vente->addPaiementClient($paiement);
            $montantPaye += (float)$paiement->getMontant();
        }

        $vente->setMontantPaye((string)$montantPaye);
        $vente->setMontantRestant((string)((float)$vente->getMontantFinal() - $montantPaye));
        $vente->mettreAJourStatutPaiement();

        // Si crédit, définir la date limite
        if ((float)$vente->getMontantRestant() > 0 && isset($data['date_limite_paiement'])) {
            $vente->setDateLimitePaiement(new \DateTime($data['date_limite_paiement']));
            if (!$vente->getClient()) {
                return new JsonResponse(['error' => 'Un client est obligatoire pour les ventes à crédit'], 400);
            }
        }

        $vente->setModePaiement($data['mode_paiement'] ?? Vente::MODE_PAIEMENT_CASH);
        $vente->setNotes($data['notes'] ?? null);

        $em->persist($vente);
        $em->flush();

        $logService->log($user, $magasin, 'Création vente', 'Vente', $vente->getId(), "Vente #{$vente->getNumeroFacture()} - Montant: {$vente->getMontantFinal()} MAD");

        return new JsonResponse([
            'success' => true,
            'vente_id' => $vente->getId(),
            'numero_facture' => $vente->getNumeroFacture(),
            'redirect' => $this->generateUrl('ventes_show', ['id' => $vente->getId()]),
        ]);
    }

    #[Route('/api/recherche-produit', name: 'api_recherche_produit', methods: ['GET'])]
    public function rechercheProduit(Request $request, ProduitRepository $produitRepository): JsonResponse
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $query = $request->query->get('q', '');
        $limit = (int)($request->query->get('limit', 10));

        if (empty($query)) {
            return new JsonResponse([]);
        }

        // Recherche par code-barres, référence, IMEI, nom
        $produits = $produitRepository->createQueryBuilder('p')
            ->where('p.magasin = :magasin')
            ->andWhere('p.actif = true')
            ->andWhere('p.statut = :statut')
            ->andWhere('(p.codeBarres LIKE :query OR p.referenceInterne LIKE :query OR p.imei LIKE :query OR p.nom LIKE :query)')
            ->setParameter('magasin', $magasin)
            ->setParameter('statut', Produit::STATUT_EN_STOCK)
            ->setParameter('query', '%' . $query . '%')
            ->setMaxResults($limit)
            ->getQuery()
            ->getResult();

        $results = [];
        foreach ($produits as $produit) {
            $results[] = [
                'id' => $produit->getId(),
                'nom' => $produit->getNom(),
                'codeBarres' => $produit->getCodeBarres(),
                'referenceInterne' => $produit->getReferenceInterne(),
                'imei' => $produit->getImei(),
                'prixAchat' => $produit->getPrixAchat(),
                'prixVenteRecommande' => $produit->getPrixVenteRecommande(),
                'quantite' => $produit->getQuantite(),
            ];
        }

        return new JsonResponse($results);
    }

    #[Route('/api/produits-sous-categorie/{sousCategorieId}', name: 'api_produits_sous_categorie', methods: ['GET'])]
    public function getProduitsSousCategorie(int $sousCategorieId, ProduitRepository $produitRepository, EntityManagerInterface $em): JsonResponse
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        // Vérifier que la sous-catégorie appartient au magasin
        $sousCategorie = $em->getRepository(\App\Entity\SousCategorie::class)->find($sousCategorieId);
        if (!$sousCategorie || $sousCategorie->getMagasin() !== $magasin) {
            return new JsonResponse(['error' => 'Sous-catégorie non trouvée'], 404);
        }

        // Récupérer les produits disponibles de cette sous-catégorie
        $produits = $produitRepository->createQueryBuilder('p')
            ->where('p.magasin = :magasin')
            ->andWhere('p.actif = true')
            ->andWhere('p.sousCategorie = :sousCategorieId')
            ->andWhere('p.statut != :statutVendu')
            ->andWhere('p.quantite > 0')
            ->setParameter('magasin', $magasin)
            ->setParameter('sousCategorieId', $sousCategorieId)
            ->setParameter('statutVendu', Produit::STATUT_VENDU)
            ->orderBy('p.nom', 'ASC')
            ->getQuery()
            ->getResult();

        $results = [];
        foreach ($produits as $produit) {
            $results[] = [
                'id' => $produit->getId(),
                'nom' => $produit->getNom(),
                'codeBarres' => $produit->getCodeBarres(),
                'referenceInterne' => $produit->getReferenceInterne(),
                'imei' => $produit->getImei(),
                'numeroSerie' => $produit->getNumeroSerie(),
                'prixAchat' => $produit->getPrixAchat(),
                'prixVenteRecommande' => $produit->getPrixVenteRecommande(),
                'margeMinimale' => $produit->getMargeMinimale(),
                'quantite' => $produit->getQuantite(),
                'statut' => $produit->getStatut(),
                'caracteristiques' => $produit->getCaracteristiquesValeurs() ?? [],
            ];
        }

        return new JsonResponse($results);
    }

    #[Route('/{id}', name: 'show', methods: ['GET'])]
    public function show(Vente $vente): Response
    {
        $user = $this->getUser();
        if ($vente->getMagasin() !== $user->getMagasin()) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('vente/show.html.twig', [
            'vente' => $vente,
        ]);
    }

    #[Route('/{id}/imprimer', name: 'imprimer', methods: ['GET'])]
    public function imprimer(Vente $vente): Response
    {
        $user = $this->getUser();
        if ($vente->getMagasin() !== $user->getMagasin()) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('vente/facture.html.twig', [
            'vente' => $vente,
        ]);
    }
}
