#!/bin/bash

# Script de déploiement pour la production
# Usage: ./deploy.sh

set -e  # Arrêter en cas d'erreur

echo "🚀 Début du déploiement..."

# Couleurs pour les messages
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Vérifier si Docker est installé
if ! command -v docker &> /dev/null; then
    echo -e "${RED}❌ Docker n'est pas installé. Veuillez l'installer d'abord.${NC}"
    exit 1
fi

# Vérifier si Docker Compose est installé
if ! command -v docker-compose &> /dev/null; then
    echo -e "${RED}❌ Docker Compose n'est pas installé. Veuillez l'installer d'abord.${NC}"
    exit 1
fi

# Vérifier si le fichier .env existe
if [ ! -f .env ]; then
    echo -e "${YELLOW}⚠️  Le fichier .env n'existe pas.${NC}"
    if [ -f .env.example ]; then
        echo -e "${YELLOW}📋 Copie de .env.example vers .env...${NC}"
        cp .env.example .env
        echo -e "${YELLOW}⚠️  Veuillez configurer le fichier .env avant de continuer.${NC}"
        exit 1
    else
        echo -e "${RED}❌ Le fichier .env.example n'existe pas non plus.${NC}"
        exit 1
    fi
fi

echo -e "${GREEN}✅ Vérifications préliminaires réussies${NC}"

# Construire les images
echo -e "${YELLOW}🔨 Construction des images Docker...${NC}"
docker-compose -f docker-compose.prod.yml build --no-cache

# Arrêter les conteneurs existants
echo -e "${YELLOW}🛑 Arrêt des conteneurs existants...${NC}"
docker-compose -f docker-compose.prod.yml down

# Démarrer les conteneurs
echo -e "${YELLOW}🚀 Démarrage des conteneurs...${NC}"
docker-compose -f docker-compose.prod.yml up -d

# Attendre que la base de données soit prête
echo -e "${YELLOW}⏳ Attente que la base de données soit prête...${NC}"
sleep 10

# Installer les dépendances
echo -e "${YELLOW}📦 Installation des dépendances...${NC}"
docker-compose -f docker-compose.prod.yml exec -T php composer install --no-dev --optimize-autoloader --no-interaction

# Créer la base de données si elle n'existe pas
echo -e "${YELLOW}💾 Création de la base de données...${NC}"
docker-compose -f docker-compose.prod.yml exec -T php php bin/console doctrine:database:create --if-not-exists || true

# Exécuter les migrations
echo -e "${YELLOW}🔄 Exécution des migrations...${NC}"
docker-compose -f docker-compose.prod.yml exec -T php php bin/console doctrine:migrations:migrate --no-interaction

# Vider le cache
echo -e "${YELLOW}🧹 Vidage du cache...${NC}"
docker-compose -f docker-compose.prod.yml exec -T php php bin/console cache:clear --env=prod --no-debug

# Réchauffer le cache
echo -e "${YELLOW}🔥 Réchauffement du cache...${NC}"
docker-compose -f docker-compose.prod.yml exec -T php php bin/console cache:warmup --env=prod

# Définir les permissions
echo -e "${YELLOW}🔐 Configuration des permissions...${NC}"
docker-compose -f docker-compose.prod.yml exec -T php chown -R www-data:www-data /var/www/html
docker-compose -f docker-compose.prod.yml exec -T php chmod -R 755 /var/www/html
docker-compose -f docker-compose.prod.yml exec -T php chmod -R 775 /var/www/html/var
docker-compose -f docker-compose.prod.yml exec -T php chmod -R 775 /var/www/html/public/uploads

# Vérifier l'état des conteneurs
echo -e "${YELLOW}📊 Vérification de l'état des conteneurs...${NC}"
docker-compose -f docker-compose.prod.yml ps

echo -e "${GREEN}✅ Déploiement terminé avec succès!${NC}"
echo -e "${GREEN}🌐 L'application devrait être accessible sur http://localhost${NC}"

# Afficher les logs
echo -e "${YELLOW}📝 Voulez-vous voir les logs? (o/n)${NC}"
read -r response
if [[ "$response" =~ ^([oO][uU][iI]|[oO])$ ]]; then
    docker-compose -f docker-compose.prod.yml logs -f
fi

