<?php

namespace App\Service;

use App\Entity\Produit;
use App\Entity\User;
use App\Entity\Magasin;
use Doctrine\ORM\EntityManagerInterface;

class HistoriqueProduitService
{
    public function __construct(
        private EntityManagerInterface $em
    ) {
    }

    /**
     * Enregistre une action dans l'historique du produit
     */
    public function enregistrerAction(
        Produit $produit,
        User $user,
        Magasin $magasin,
        string $action,
        ?string $details = null
    ): void {
        // Pour l'instant, on utilise le LogService existant
        // Plus tard, on pourra créer une entité HistoriqueProduit dédiée
        // qui stockera l'historique complet avec plus de détails
        
        $message = sprintf(
            '[%s] %s',
            (new \DateTime())->format('d/m/Y H:i'),
            $action
        );
        
        if ($details) {
            $message .= ' : ' . $details;
        }
        
        // On pourrait stocker cela dans une table dédiée ou dans un champ JSON du produit
        // Pour l'instant, on utilise le système de logs existant
    }

    /**
     * Récupère l'historique d'un produit
     */
    public function getHistorique(Produit $produit): array
    {
        $historique = [];
        
        // Création
        $historique[] = [
            'date' => $produit->getDateEntree(),
            'action' => 'Création du produit',
            'details' => sprintf('Créé par le système le %s', $produit->getDateEntree()->format('d/m/Y à H:i'))
        ];
        
        // Achat
        if ($produit->getDateAchat()) {
            if ($produit->getOrigineAchat() === Produit::ORIGINE_FOURNISSEUR && $produit->getFournisseur()) {
                $historique[] = [
                    'date' => $produit->getDateAchat(),
                    'action' => 'Achat fournisseur',
                    'details' => sprintf(
                        'Acheté auprès de %s pour %s MAD',
                        $produit->getFournisseur()->getNom(),
                        number_format((float)$produit->getPrixAchat(), 2, ',', ' ')
                    )
                ];
            } elseif ($produit->getOrigineAchat() === Produit::ORIGINE_PARTICULIER && $produit->getAchatParticulier()) {
                $historique[] = [
                    'date' => $produit->getDateAchat(),
                    'action' => 'Achat particulier',
                    'details' => sprintf(
                        'Acheté auprès de %s pour %s MAD',
                        $produit->getAchatParticulier()->getNomClient(),
                        number_format((float)$produit->getPrixAchat(), 2, ',', ' ')
                    )
                ];
            }
        }
        
        // Changements de statut (à implémenter avec une table d'historique dédiée)
        // Pour l'instant, on retourne juste les événements de base
        
        // Trier par date décroissante
        usort($historique, function($a, $b) {
            return $b['date'] <=> $a['date'];
        });
        
        return $historique;
    }
}

