<?php

namespace App\Service;

use App\Entity\Magasin;
use App\Entity\User;
use App\Repository\ProduitRepository;
use App\Repository\VenteRepository;
use App\Repository\AchatRepository;
use App\Repository\DepenseRepository;
use App\Repository\ClientRepository;
use App\Repository\PaiementClientRepository;
use App\Repository\FournisseurRepository;
use App\Repository\PaiementRepository;
use App\Repository\AchatParticulierRepository;
use Doctrine\ORM\EntityManagerInterface;

class DashboardService
{
    public function __construct(
        private ProduitRepository $produitRepository,
        private VenteRepository $venteRepository,
        private AchatRepository $achatRepository,
        private DepenseRepository $depenseRepository,
        private ClientRepository $clientRepository,
        private PaiementClientRepository $paiementClientRepository,
        private FournisseurRepository $fournisseurRepository,
        private PaiementRepository $paiementRepository,
        private AchatParticulierRepository $achatParticulierRepository,
        private EntityManagerInterface $em
    ) {
    }

    /**
     * Calcule les dates de début et fin selon la période sélectionnée
     */
    public function getDatesFromPeriod(string $period, ?\DateTime $dateDebutCustom = null, ?\DateTime $dateFinCustom = null): array
    {
        $now = new \DateTime();
        $dateFin = $dateFinCustom ?? clone $now;
        
        switch ($period) {
            case 'today':
                $dateDebut = (clone $now)->setTime(0, 0, 0);
                break;
            case 'yesterday':
                $dateDebut = (clone $now)->modify('-1 day')->setTime(0, 0, 0);
                $dateFin = (clone $now)->modify('-1 day')->setTime(23, 59, 59);
                break;
            case 'this_week':
                $dateDebut = (clone $now)->modify('monday this week')->setTime(0, 0, 0);
                break;
            case 'this_month':
                $dateDebut = (clone $now)->modify('first day of this month')->setTime(0, 0, 0);
                break;
            case 'this_quarter':
                $month = (int)$now->format('n');
                $quarter = (int)ceil($month / 3);
                $dateDebut = (clone $now)->modify("first day of january")->modify("+".(($quarter-1)*3)." months")->setTime(0, 0, 0);
                break;
            case 'this_year':
                $dateDebut = (clone $now)->modify('first day of january')->setTime(0, 0, 0);
                break;
            case 'custom':
                $dateDebut = $dateDebutCustom ?? (clone $now)->modify('first day of this month')->setTime(0, 0, 0);
                break;
            default:
                $dateDebut = (clone $now)->modify('first day of this month')->setTime(0, 0, 0);
        }
        
        $dateFin->setTime(23, 59, 59);
        
        return [$dateDebut, $dateFin];
    }

    /**
     * Récupère les magasins selon le rôle de l'utilisateur
     */
    public function getMagasinsForUser(User $user): array
    {
        if (in_array('ROLE_SUPER_ADMIN', $user->getRoles())) {
            // Super admin voit tous les magasins
            return $this->em->getRepository(\App\Entity\Magasin::class)->findBy(['actif' => true]);
        }
        
        // Sinon, seulement le magasin de l'utilisateur
        return [$user->getMagasin()];
    }

    /**
     * Récupère les utilisateurs selon les filtres
     */
    public function getUsersForFilter(?Magasin $magasin, ?int $userId): array
    {
        if ($userId) {
            $user = $this->em->getRepository(\App\Entity\User::class)->find($userId);
            return $user ? [$user] : [];
        }
        
        if ($magasin) {
            return $magasin->getUsers()->filter(fn($u) => $u->isActif())->toArray();
        }
        
        return [];
    }

    /**
     * Calcule les KPIs principaux
     */
    public function getKPIs(array $magasins, ?\DateTime $dateDebut, ?\DateTime $dateFin, ?User $user = null, ?int $categorieId = null): array
    {
        $dateDebut = $dateDebut ?? new \DateTime('first day of this month');
        $dateFin = $dateFin ?? new \DateTime('now');
        $dateFin->setTime(23, 59, 59);

        // Ventes
        $ventes = $this->getVentes($magasins, $dateDebut, $dateFin, $user, $categorieId);
        $totalVentes = array_sum(array_map(fn($v) => (float)$v->getMontantFinal(), $ventes));
        $nombreProduitsVendus = $this->getNombreProduitsVendus($ventes);
        $profit = $this->getProfit($ventes);
        $margeMoyenne = count($ventes) > 0 ? $profit / count($ventes) : 0;
        
        // Paiements reçus
        $paiementsRecus = $this->getPaiementsRecus($magasins, $dateDebut, $dateFin, $user);
        
        // Dettes clients
        $dettesClients = $this->getDettesClients($magasins);
        $nombreClientsCredit = $this->getNombreClientsAvecCredit($magasins);
        
        // Dettes fournisseurs
        $dettesFournisseurs = $this->getDettesFournisseurs($magasins);
        $nombreFacturesOuvertes = $this->getNombreFacturesOuvertes($magasins);
        
        // Stock
        $stockData = $this->getStockData($magasins);
        
        // Calcul période précédente pour évolution
        $periodePrecedente = $this->getPeriodePrecedente($dateDebut, $dateFin);
        $ventesPrecedentes = $this->getVentes($magasins, $periodePrecedente[0], $periodePrecedente[1], $user, $categorieId);
        $totalVentesPrecedentes = array_sum(array_map(fn($v) => (float)$v->getMontantFinal(), $ventesPrecedentes));
        $evolutionVentes = $totalVentesPrecedentes > 0 
            ? (($totalVentes - $totalVentesPrecedentes) / $totalVentesPrecedentes) * 100 
            : 0;

        return [
            'totalVentes' => $totalVentes,
            'evolutionVentes' => $evolutionVentes,
            'nombreProduitsVendus' => $nombreProduitsVendus,
            'profit' => $profit,
            'margeMoyenne' => $margeMoyenne,
            'paiementsRecus' => $paiementsRecus,
            'dettesClients' => $dettesClients,
            'nombreClientsCredit' => $nombreClientsCredit,
            'dettesFournisseurs' => $dettesFournisseurs,
            'nombreFacturesOuvertes' => $nombreFacturesOuvertes,
            'valeurStock' => $stockData['valeur'],
            'produitsEnStock' => $stockData['total'],
            'produitsNeufs' => $stockData['neuf'],
            'produitsOccasion' => $stockData['occasion'],
        ];
    }

    /**
     * Récupère les ventes selon les filtres
     */
    private function getVentes(array $magasins, \DateTime $dateDebut, \DateTime $dateFin, ?User $user = null, ?int $categorieId = null): array
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('v')
            ->from(\App\Entity\Vente::class, 'v')
            ->where('v.magasin IN (:magasins)')
            ->andWhere('v.dateVente BETWEEN :dateDebut AND :dateFin')
            ->andWhere('v.retour = false')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin);

        if ($user) {
            $qb->andWhere('v.utilisateur = :user')
                ->setParameter('user', $user);
        }

        if ($categorieId) {
            $qb->join('v.lignesVente', 'lv')
                ->join('lv.produit', 'p')
                ->andWhere('p.categorie = :categorieId')
                ->setParameter('categorieId', $categorieId);
        }

        return $qb->getQuery()->getResult();
    }

    private function getNombreProduitsVendus(array $ventes): int
    {
        $total = 0;
        foreach ($ventes as $vente) {
            foreach ($vente->getLignesVente() as $ligne) {
                $total += $ligne->getQuantite();
            }
        }
        return $total;
    }

    private function getProfit(array $ventes): float
    {
        $profit = 0;
        foreach ($ventes as $vente) {
            $profit += $vente->getMargeTotale();
        }
        return $profit;
    }

    private function getPaiementsRecus(array $magasins, \DateTime $dateDebut, \DateTime $dateFin, ?User $user = null): float
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('SUM(pc.montant) as total')
            ->from(\App\Entity\PaiementClient::class, 'pc')
            ->join('pc.vente', 'v')
            ->where('v.magasin IN (:magasins)')
            ->andWhere('pc.datePaiement BETWEEN :dateDebut AND :dateFin')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin);

        if ($user) {
            $qb->andWhere('pc.utilisateur = :user')
                ->setParameter('user', $user);
        }

        $result = $qb->getQuery()->getSingleScalarResult();
        return (float)($result ?? 0);
    }

    private function getDettesClients(array $magasins): float
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('SUM(v.montantRestant) as total')
            ->from(\App\Entity\Vente::class, 'v')
            ->where('v.magasin IN (:magasins)')
            ->andWhere('v.retour = false')
            ->andWhere('v.montantRestant > 0')
            ->setParameter('magasins', $magasins);

        $result = $qb->getQuery()->getSingleScalarResult();
        return (float)($result ?? 0);
    }

    private function getNombreClientsAvecCredit(array $magasins): int
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('COUNT(DISTINCT v.client) as total')
            ->from(\App\Entity\Vente::class, 'v')
            ->where('v.magasin IN (:magasins)')
            ->andWhere('v.retour = false')
            ->andWhere('v.montantRestant > 0')
            ->andWhere('v.client IS NOT NULL')
            ->setParameter('magasins', $magasins);

        $result = $qb->getQuery()->getSingleScalarResult();
        return (int)($result ?? 0);
    }

    private function getDettesFournisseurs(array $magasins): float
    {
        $total = 0;
        foreach ($magasins as $magasin) {
            foreach ($magasin->getFournisseurs() as $fournisseur) {
                if (!$fournisseur->isActif()) continue;
                
                $achats = $this->achatRepository->findBy(['fournisseur' => $fournisseur]);
                foreach ($achats as $achat) {
                    $totalPaye = $this->paiementRepository->getTotalPayeByAchat($achat->getId());
                    $reste = (float)$achat->getMontantTotal() - $totalPaye;
                    if ($reste > 0) {
                        $total += $reste;
                    }
                }
            }
        }
        return $total;
    }

    private function getNombreFacturesOuvertes(array $magasins): int
    {
        $count = 0;
        foreach ($magasins as $magasin) {
            foreach ($magasin->getFournisseurs() as $fournisseur) {
                if (!$fournisseur->isActif()) continue;
                
                $achats = $this->achatRepository->findBy(['fournisseur' => $fournisseur]);
                foreach ($achats as $achat) {
                    $totalPaye = $this->paiementRepository->getTotalPayeByAchat($achat->getId());
                    $reste = (float)$achat->getMontantTotal() - $totalPaye;
                    if ($reste > 0) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }

    private function getStockData(array $magasins): array
    {
        $total = 0;
        $valeur = 0.0;
        $neuf = 0;
        $occasion = 0;

        foreach ($magasins as $magasin) {
            $produits = $this->produitRepository->findBy(['magasin' => $magasin, 'actif' => true]);
            foreach ($produits as $produit) {
                if ($produit->getStatut() === \App\Entity\Produit::STATUT_VENDU) {
                    continue;
                }
                
                $total += $produit->getQuantite();
                $valeur += $produit->getQuantite() * (float)$produit->getPrixAchat();
                
                // Détecter neuf/occasion via caractéristiques (à adapter selon votre structure)
                $caracteristiques = $produit->getCaracteristiquesValeurs() ?? [];
                // Logique à adapter selon vos caractéristiques
            }
        }

        return [
            'total' => $total,
            'valeur' => $valeur,
            'neuf' => $neuf,
            'occasion' => $occasion,
        ];
    }

    private function getPeriodePrecedente(\DateTime $dateDebut, \DateTime $dateFin): array
    {
        $duree = $dateDebut->diff($dateFin);
        $dateFinPrecedente = (clone $dateDebut)->modify('-1 day');
        $dateDebutPrecedente = (clone $dateFinPrecedente)->modify('-' . $duree->days . ' days');
        
        return [$dateDebutPrecedente, $dateFinPrecedente];
    }

    /**
     * Récupère les alertes
     */
    public function getAlertes(array $magasins): array
    {
        $alertes = [];
        
        // Produits avec marge faible
        foreach ($magasins as $magasin) {
            $produits = $this->produitRepository->findBy(['magasin' => $magasin, 'actif' => true]);
            foreach ($produits as $produit) {
                if ($produit->getPrixVenteRecommande() && $produit->getPrixAchat()) {
                    $marge = (float)$produit->getPrixVenteRecommande() - (float)$produit->getPrixAchat();
                    $margeMinimale = (float)($produit->getMargeMinimale() ?? 0);
                    if ($marge < $margeMinimale) {
                        $alertes[] = [
                            'type' => 'marge_faible',
                            'severite' => 'warning',
                            'message' => "Produit '{$produit->getNom()}' a une marge inférieure à la marge minimale",
                            'produit' => $produit,
                        ];
                    }
                }
            }
        }
        
        // Crédits expirés
        $qb = $this->em->createQueryBuilder();
        $qb->select('v')
            ->from(\App\Entity\Vente::class, 'v')
            ->where('v.magasin IN (:magasins)')
            ->andWhere('v.retour = false')
            ->andWhere('v.montantRestant > 0')
            ->andWhere('v.dateLimitePaiement IS NOT NULL')
            ->andWhere('v.dateLimitePaiement < :now')
            ->setParameter('magasins', $magasins)
            ->setParameter('now', new \DateTime());
        
        $creditsExpires = $qb->getQuery()->getResult();
        foreach ($creditsExpires as $vente) {
            $alertes[] = [
                'type' => 'credit_expire',
                'severite' => 'danger',
                'message' => "Crédit expiré pour la vente #{$vente->getNumeroFacture()} - Client: " . ($vente->getClient() ? $vente->getClient()->getNom() : 'N/A'),
                'vente' => $vente,
            ];
        }
        
        // Dettes fournisseur arrivant sous 48h
        $dateLimite = (new \DateTime())->modify('+2 days');
        foreach ($magasins as $magasin) {
            foreach ($magasin->getFournisseurs() as $fournisseur) {
                if (!$fournisseur->isActif()) continue;
                
                $achats = $this->achatRepository->findBy(['fournisseur' => $fournisseur]);
                foreach ($achats as $achat) {
                    if ($achat->getDateLimitePaiement() && $achat->getDateLimitePaiement() <= $dateLimite) {
                        $totalPaye = $this->paiementRepository->getTotalPayeByAchat($achat->getId());
                        $reste = (float)$achat->getMontantTotal() - $totalPaye;
                        if ($reste > 0) {
                            $alertes[] = [
                                'type' => 'dette_fournisseur',
                                'severite' => 'warning',
                                'message' => "Dette fournisseur '{$fournisseur->getNom()}' arrive à échéance le " . $achat->getDateLimitePaiement()->format('d/m/Y'),
                                'achat' => $achat,
                            ];
                        }
                    }
                }
            }
        }
        
        // Produits en stock depuis + 30 jours
        $dateLimiteStock = (new \DateTime())->modify('-30 days');
        foreach ($magasins as $magasin) {
            $produits = $this->produitRepository->findBy(['magasin' => $magasin, 'actif' => true]);
            foreach ($produits as $produit) {
                if ($produit->getDateEntree() && $produit->getDateEntree() < $dateLimiteStock) {
                    if ($produit->getStatut() !== \App\Entity\Produit::STATUT_VENDU) {
                        $alertes[] = [
                            'type' => 'stock_ancien',
                            'severite' => 'info',
                            'message' => "Produit '{$produit->getNom()}' en stock depuis plus de 30 jours",
                            'produit' => $produit,
                        ];
                    }
                }
            }
        }
        
        return $alertes;
    }

    /**
     * Récupère les données pour les graphiques
     */
    public function getGraphiques(array $magasins, \DateTime $dateDebut, \DateTime $dateFin, ?User $user = null): array
    {
        return [
            'ventesParPeriode' => $this->getVentesParPeriode($magasins, $dateDebut, $dateFin, $user),
            'ventesParCategorie' => $this->getVentesParCategorie($magasins, $dateDebut, $dateFin, $user),
            'topProduits' => $this->getTopProduits($magasins, $dateDebut, $dateFin, $user),
            'provenanceAchats' => $this->getProvenanceAchats($magasins, $dateDebut, $dateFin),
            'paiementsParMode' => $this->getPaiementsParMode($magasins, $dateDebut, $dateFin, $user),
            'depensesParType' => $this->getDepensesParType($magasins, $dateDebut, $dateFin),
            'rentabilite' => $this->getRentabilite($magasins, $dateDebut, $dateFin),
        ];
    }

    private function getVentesParPeriode(array $magasins, \DateTime $dateDebut, \DateTime $dateFin, ?User $user = null): array
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('v')
            ->from(\App\Entity\Vente::class, 'v')
            ->where('v.magasin IN (:magasins)')
            ->andWhere('v.dateVente BETWEEN :dateDebut AND :dateFin')
            ->andWhere('v.retour = false')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->orderBy('v.dateVente', 'ASC');

        if ($user) {
            $qb->andWhere('v.utilisateur = :user')
                ->setParameter('user', $user);
        }

        $ventes = $qb->getQuery()->getResult();
        
        // Grouper par date en PHP (compatible avec toutes les bases de données)
        $grouped = [];
        foreach ($ventes as $vente) {
            $date = $vente->getDateVente();
            if ($date instanceof \DateTime) {
                $dateKey = $date->format('Y-m-d');
            } else {
                $dateKey = (new \DateTime($date))->format('Y-m-d');
            }
            
            if (!isset($grouped[$dateKey])) {
                $grouped[$dateKey] = [
                    'date' => $dateKey,
                    'total' => 0,
                    'nombre' => 0
                ];
            }
            
            $grouped[$dateKey]['total'] += (float)$vente->getMontantFinal();
            $grouped[$dateKey]['nombre'] += 1;
        }
        
        return array_values($grouped);
    }

    private function getVentesParCategorie(array $magasins, \DateTime $dateDebut, \DateTime $dateFin, ?User $user = null): array
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('c.nom as categorie, SUM(lv.montantTotal) as total')
            ->from(\App\Entity\LigneVente::class, 'lv')
            ->join('lv.vente', 'v')
            ->join('lv.produit', 'p')
            ->join('p.categorie', 'c')
            ->where('v.magasin IN (:magasins)')
            ->andWhere('v.dateVente BETWEEN :dateDebut AND :dateFin')
            ->andWhere('v.retour = false')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->groupBy('c.id', 'c.nom')
            ->orderBy('total', 'DESC');

        if ($user) {
            $qb->andWhere('v.utilisateur = :user')
                ->setParameter('user', $user);
        }

        return $qb->getQuery()->getResult();
    }

    private function getTopProduits(array $magasins, \DateTime $dateDebut, \DateTime $dateFin, ?User $user = null, int $limit = 10): array
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('p.id, p.nom, SUM(lv.quantite) as quantite, SUM(lv.montantTotal) as total, SUM((lv.prixUnitaire - lv.prixAchatUnitaire) * lv.quantite) as profit')
            ->from(\App\Entity\LigneVente::class, 'lv')
            ->join('lv.vente', 'v')
            ->join('lv.produit', 'p')
            ->where('v.magasin IN (:magasins)')
            ->andWhere('v.dateVente BETWEEN :dateDebut AND :dateFin')
            ->andWhere('v.retour = false')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->groupBy('p.id', 'p.nom')
            ->orderBy('total', 'DESC')
            ->setMaxResults($limit);

        if ($user) {
            $qb->andWhere('v.utilisateur = :user')
                ->setParameter('user', $user);
        }

        return $qb->getQuery()->getResult();
    }

    private function getProvenanceAchats(array $magasins, \DateTime $dateDebut, \DateTime $dateFin): array
    {
        // Achats fournisseurs
        $qb1 = $this->em->createQueryBuilder();
        $qb1->select('SUM(a.montantTotal) as total')
            ->from(\App\Entity\Achat::class, 'a')
            ->where('a.magasin IN (:magasins)')
            ->andWhere('a.dateAchat BETWEEN :dateDebut AND :dateFin')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin);
        
        $achatsFournisseurs = (float)($qb1->getQuery()->getSingleScalarResult() ?? 0);
        
        // Achats particuliers
        $qb2 = $this->em->createQueryBuilder();
        $qb2->select('SUM(ap.prixPaye) as total')
            ->from(\App\Entity\AchatParticulier::class, 'ap')
            ->join('ap.produit', 'p')
            ->where('p.magasin IN (:magasins)')
            ->andWhere('ap.dateAchat BETWEEN :dateDebut AND :dateFin')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin);
        
        $achatsParticuliers = (float)($qb2->getQuery()->getSingleScalarResult() ?? 0);
        
        return [
            'fournisseurs' => $achatsFournisseurs,
            'particuliers' => $achatsParticuliers,
        ];
    }

    private function getPaiementsParMode(array $magasins, \DateTime $dateDebut, \DateTime $dateFin, ?User $user = null): array
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('v.modePaiement, SUM(pc.montant) as total')
            ->from(\App\Entity\PaiementClient::class, 'pc')
            ->join('pc.vente', 'v')
            ->where('v.magasin IN (:magasins)')
            ->andWhere('pc.datePaiement BETWEEN :dateDebut AND :dateFin')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->groupBy('v.modePaiement');

        if ($user) {
            $qb->andWhere('pc.utilisateur = :user')
                ->setParameter('user', $user);
        }

        $result = $qb->getQuery()->getResult();
        
        // Ajouter les crédits (montant restant)
        $credits = $this->getDettesClients($magasins);
        
        $modes = [];
        foreach ($result as $row) {
            $modes[$row['modePaiement']] = (float)$row['total'];
        }
        $modes['Crédit'] = $credits;
        
        return $modes;
    }

    private function getDepensesParType(array $magasins, \DateTime $dateDebut, \DateTime $dateFin): array
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('d.categorie, SUM(d.montant) as total')
            ->from(\App\Entity\Depense::class, 'd')
            ->where('d.magasin IN (:magasins)')
            ->andWhere('d.dateDepense BETWEEN :dateDebut AND :dateFin')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->groupBy('d.categorie')
            ->orderBy('total', 'DESC');

        $result = $qb->getQuery()->getResult();
        
        $depenses = [];
        foreach ($result as $row) {
            $depenses[$row['categorie']] = (float)$row['total'];
        }
        
        return $depenses;
    }

    private function getRentabilite(array $magasins, \DateTime $dateDebut, \DateTime $dateFin): array
    {
        // Ventes
        $ventes = $this->getVentes($magasins, $dateDebut, $dateFin);
        $totalVentes = array_sum(array_map(fn($v) => (float)$v->getMontantFinal(), $ventes));
        
        // Dépenses
        $qb = $this->em->createQueryBuilder();
        $qb->select('SUM(d.montant) as total')
            ->from(\App\Entity\Depense::class, 'd')
            ->where('d.magasin IN (:magasins)')
            ->andWhere('d.dateDepense BETWEEN :dateDebut AND :dateFin')
            ->setParameter('magasins', $magasins)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin);
        
        $totalDepenses = (float)($qb->getQuery()->getSingleScalarResult() ?? 0);
        
        // Profit
        $profit = $this->getProfit($ventes);
        $resultatNet = $profit - $totalDepenses;
        
        return [
            'ventes' => $totalVentes,
            'depenses' => $totalDepenses,
            'profit' => $profit,
            'resultatNet' => $resultatNet,
        ];
    }
}

