<?php

namespace App\Entity;

use App\Repository\CaracteristiqueRepository;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use App\Validator\Constraints as AppAssert;

#[ORM\Entity(repositoryClass: CaracteristiqueRepository::class)]
#[ORM\Table(name: 'caracteristiques')]
#[AppAssert\CaracteristiqueValide]
class Caracteristique
{
    const TYPE_TEXT = 'text';
    const TYPE_NUMBER = 'number';
    const TYPE_BOOLEAN = 'boolean';
    const TYPE_SELECT = 'select';
    const TYPE_MULTI_SELECT = 'multi-select';

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 255)]
    private ?string $nom = null;

    #[ORM\Column(length: 50)]
    private ?string $type = self::TYPE_TEXT;

    #[ORM\Column(type: Types::JSON, nullable: true)]
    private ?array $valeursPossibles = null; // Pour type select/multi-select, tableau JSON

    #[ORM\ManyToOne(inversedBy: 'caracteristiques')]
    #[ORM\JoinColumn(nullable: true)]
    private ?Categorie $categorie = null;

    #[ORM\ManyToOne(inversedBy: 'caracteristiques')]
    #[ORM\JoinColumn(nullable: true)]
    private ?SousCategorie $sousCategorie = null;

    #[ORM\ManyToOne(inversedBy: 'caracteristiques')]
    #[ORM\JoinColumn(nullable: false)]
    private ?Magasin $magasin = null;

    #[ORM\Column(type: Types::BOOLEAN)]
    private bool $obligatoire = false;

    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    private ?\DateTimeInterface $dateCreation = null;

    public function __construct()
    {
        $this->dateCreation = new \DateTime();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getNom(): ?string
    {
        return $this->nom;
    }

    public function setNom(string $nom): self
    {
        $this->nom = $nom;
        return $this;
    }

    public function getType(): ?string
    {
        return $this->type;
    }

    public function setType(string $type): self
    {
        $this->type = $type;
        return $this;
    }

    public function getValeursPossibles(): ?array
    {
        return $this->valeursPossibles ?? [];
    }

    public function setValeursPossibles(?array $valeursPossibles): self
    {
        $this->valeursPossibles = $valeursPossibles;
        return $this;
    }

    public function getCategorie(): ?Categorie
    {
        return $this->categorie;
    }

    public function setCategorie(?Categorie $categorie): self
    {
        $this->categorie = $categorie;
        return $this;
    }

    public function isObligatoire(): bool
    {
        return $this->obligatoire;
    }

    public function setObligatoire(bool $obligatoire): self
    {
        $this->obligatoire = $obligatoire;
        return $this;
    }

    public function getSousCategorie(): ?SousCategorie
    {
        return $this->sousCategorie;
    }

    public function setSousCategorie(?SousCategorie $sousCategorie): self
    {
        $this->sousCategorie = $sousCategorie;
        return $this;
    }

    public function getMagasin(): ?Magasin
    {
        return $this->magasin;
    }

    public function setMagasin(?Magasin $magasin): self
    {
        $this->magasin = $magasin;
        return $this;
    }

    /**
     * Vérifie si la caractéristique nécessite des valeurs possibles
     */
    public function requiresValeursPossibles(): bool
    {
        return in_array($this->type, [self::TYPE_SELECT, self::TYPE_MULTI_SELECT]);
    }

    public function getDateCreation(): ?\DateTimeInterface
    {
        return $this->dateCreation;
    }

    public function setDateCreation(\DateTimeInterface $dateCreation): self
    {
        $this->dateCreation = $dateCreation;
        return $this;
    }

    /**
     * Vérifie que la caractéristique appartient soit à une catégorie soit à une sous-catégorie (jamais les deux)
     */
    public function isValid(): bool
    {
        return ($this->categorie !== null && $this->sousCategorie === null) 
            || ($this->categorie === null && $this->sousCategorie !== null);
    }
}

