<?php

namespace App\Controller;

use App\Entity\Paiement;
use App\Entity\Achat;
use App\Form\PaiementType;
use App\Repository\PaiementRepository;
use App\Service\LogService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/paiements', name: 'paiements_')]
#[IsGranted('ROLE_USER')]
class PaiementController extends AbstractController
{
    #[Route('/new/{achatId}', name: 'new', methods: ['GET', 'POST'])]
    #[IsGranted('ROLE_MANAGER')]
    public function new(Request $request, int $achatId, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $achat = $em->getRepository(Achat::class)->find($achatId);
        if (!$achat || $achat->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        $paiement = new Paiement();
        $paiement->setFournisseur($achat->getFournisseur());
        $paiement->setAchat($achat);
        $paiement->setUtilisateur($user);

        $form = $this->createForm(PaiementType::class, $paiement, ['achat' => $achat]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Mettre à jour le montant restant de l'achat
            $montantPaye = $achat->getMontantPaye() + (float)$paiement->getMontant();
            $montantTotal = (float)$achat->getMontantTotal();
            $nouveauMontantRestant = max(0, $montantTotal - $montantPaye);
            $achat->setMontantRestant((string)$nouveauMontantRestant);

            $em->persist($paiement);
            $em->flush();

            $logService->log(
                $user,
                $magasin,
                'Enregistrement paiement',
                'Paiement',
                $paiement->getId(),
                sprintf(
                    "Paiement de %s MAD pour l'achat %s (Fournisseur: %s)",
                    number_format((float)$paiement->getMontant(), 2, ',', ' '),
                    $achat->getNumeroFacture(),
                    $achat->getFournisseur()->getNom()
                )
            );

            $this->addFlash('success', 'Paiement enregistré avec succès');
            return $this->redirectToRoute('fournisseurs_show', ['id' => $achat->getFournisseur()->getId()]);
        }

        return $this->render('paiement/new.html.twig', [
            'paiement' => $paiement,
            'achat' => $achat,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/delete', name: 'delete', methods: ['POST'])]
    #[IsGranted('ROLE_MANAGER')]
    public function delete(Request $request, Paiement $paiement, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($paiement->getFournisseur()->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        if ($this->isCsrfTokenValid('delete' . $paiement->getId(), $request->request->get('_token'))) {
            $achat = $paiement->getAchat();
            $fournisseur = $paiement->getFournisseur();

            // Recalculer le montant restant
            $montantPaye = $achat->getMontantPaye() - (float)$paiement->getMontant();
            $montantTotal = (float)$achat->getMontantTotal();
            $nouveauMontantRestant = max(0, $montantTotal - $montantPaye);
            $achat->setMontantRestant((string)$nouveauMontantRestant);

            $em->remove($paiement);
            $em->flush();

            $logService->log(
                $user,
                $magasin,
                'Suppression paiement',
                'Paiement',
                $paiement->getId(),
                sprintf("Suppression du paiement de %s MAD", number_format((float)$paiement->getMontant(), 2, ',', ' '))
            );

            $this->addFlash('success', 'Paiement supprimé avec succès');
        }

        return $this->redirectToRoute('fournisseurs_show', ['id' => $paiement->getFournisseur()->getId()]);
    }
}

