<?php

namespace App\Command;

use App\Entity\Produit;
use App\Service\CodeBarresService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

#[AsCommand(
    name: 'app:regenerer-produits',
    description: 'Vide et régénère les produits avec la nouvelle structure (quantité = 1 pour chaque produit)',
)]
class RegenererProduitsCommand extends Command
{
    public function __construct(
        private EntityManagerInterface $em,
        private CodeBarresService $codeBarresService
    ) {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->addOption('magasin-id', null, InputOption::VALUE_OPTIONAL, 'ID du magasin (si non spécifié, tous les magasins)')
            ->addOption('force', null, InputOption::VALUE_NONE, 'Forcer la suppression sans confirmation');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        
        $magasinId = $input->getOption('magasin-id');
        $force = $input->getOption('force');

        if (!$force) {
            $io->warning('Cette commande va SUPPRIMER TOUS LES PRODUITS et les régénérer avec la nouvelle structure.');
            $io->warning('Chaque produit avec quantité > 1 sera divisé en plusieurs produits distincts (quantité = 1 chacun).');
            
            if (!$io->confirm('Êtes-vous sûr de vouloir continuer ?', false)) {
                $io->info('Opération annulée.');
                return Command::SUCCESS;
            }
        }

        $io->title('Régénération des produits');

        // Récupérer tous les produits
        $qb = $this->em->getRepository(Produit::class)->createQueryBuilder('p');
        if ($magasinId) {
            $qb->where('p.magasin = :magasinId')
               ->setParameter('magasinId', $magasinId);
        }
        $produits = $qb->getQuery()->getResult();

        $io->progressStart(count($produits));
        $produitsCrees = 0;
        $produitsSupprimes = 0;

        foreach ($produits as $produit) {
            $quantite = $produit->getQuantite();
            $magasin = $produit->getMagasin();
            
            // Si quantité > 1, créer plusieurs produits
            if ($quantite > 1) {
                $codesBarresUtilises = [];
                if ($produit->getCodeBarres()) {
                    $codesBarresUtilises[] = $produit->getCodeBarres();
                }
                
                // Créer les produits supplémentaires
                for ($i = 1; $i < $quantite; $i++) {
                    $nouveauProduit = clone $produit;
                    
                    // Réinitialiser l'ID pour que Doctrine crée un nouvel enregistrement
                    $reflection = new \ReflectionClass($nouveauProduit);
                    $idProperty = $reflection->getProperty('id');
                    $idProperty->setAccessible(true);
                    $idProperty->setValue($nouveauProduit, null);
                    
                    // Générer un nouveau code-barres unique (en excluant ceux déjà utilisés)
                    $codeBarres = $this->codeBarresService->genererCodeBarres($codesBarresUtilises);
                    $nouveauProduit->setCodeBarres($codeBarres);
                    $nouveauProduit->setReferenceInterne($this->codeBarresService->genererReferenceInterne($codeBarres));
                    
                    // Ajouter le nouveau code-barres à la liste
                    $codesBarresUtilises[] = $codeBarres;
                    
                    // Réinitialiser les relations qui ne peuvent pas être clonées
                    if ($nouveauProduit->getAchatParticulier()) {
                        $nouveauProduit->setAchatParticulier(null);
                    }
                    
                    $this->em->persist($nouveauProduit);
                    $produitsCrees++;
                }
                
                // Le produit original reste tel quel (quantité déjà gérée)
                $this->em->persist($produit);
            }
            
            $io->progressAdvance();
        }

        $this->em->flush();
        $io->progressFinish();

        $io->success([
            sprintf('Régénération terminée !'),
            sprintf('Produits traités: %d', count($produits)),
            sprintf('Nouveaux produits créés: %d', $produitsCrees),
        ]);

        return Command::SUCCESS;
    }
}
