# Guide de Déploiement SANS Docker

Ce guide vous explique comment déployer l'application directement sur un serveur sans utiliser Docker.

## Prérequis

- Serveur Linux (Ubuntu 20.04+ ou Debian 11+ recommandé)
- PHP 8.2 ou supérieur
- MySQL 8.0 ou supérieur
- Nginx ou Apache
- Composer
- Accès root ou utilisateur avec droits sudo

## 1. Installation des dépendances système

### Ubuntu/Debian

```bash
# Mettre à jour le système
sudo apt update && sudo apt upgrade -y

# Installer PHP et extensions nécessaires
sudo apt install -y \
    php8.2 \
    php8.2-fpm \
    php8.2-mysql \
    php8.2-mbstring \
    php8.2-xml \
    php8.2-curl \
    php8.2-zip \
    php8.2-gd \
    php8.2-bcmath \
    php8.2-intl \
    php8.2-opcache

# Installer MySQL
sudo apt install -y mysql-server

# Installer Nginx
sudo apt install -y nginx

# Installer Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer
sudo chmod +x /usr/local/bin/composer

# Vérifier les installations
php -v
mysql --version
nginx -v
composer --version
```

## 2. Configuration de MySQL

```bash
# Sécuriser l'installation MySQL
sudo mysql_secure_installation

# Se connecter à MySQL
sudo mysql -u root -p

# Créer la base de données et l'utilisateur
CREATE DATABASE saasinfo CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'saasinfo'@'localhost' IDENTIFIED BY 'votre-mot-de-passe-securise';
GRANT ALL PRIVILEGES ON saasinfo.* TO 'saasinfo'@'localhost';
FLUSH PRIVILEGES;
EXIT;
```

## 3. Cloner et configurer l'application

```bash
# Créer le répertoire web
sudo mkdir -p /var/www
cd /var/www

# Cloner le projet (remplacer par votre URL)
git clone <votre-repository-url> facture
cd facture

# Installer les dépendances Composer
composer install --no-dev --optimize-autoloader

# Créer le fichier .env
cp ENV_EXAMPLE.txt .env
nano .env
```

### Configuration du fichier .env

```env
APP_ENV=prod
APP_SECRET=votre-secret-aleatoire-très-long-et-sécurisé
APP_DEBUG=false

DATABASE_URL="mysql://saasinfo:votre-mot-de-passe-securise@127.0.0.1:3306/saasinfo?serverVersion=8.0"

JWT_SECRET_KEY=%kernel.project_dir%/config/jwt/private.pem
JWT_PUBLIC_KEY=%kernel.project_dir%/config/jwt/public.pem
JWT_PASSPHRASE=votre-passphrase-jwt-securise
```

### Générer les secrets

```bash
# Générer APP_SECRET
php -r "echo bin2hex(random_bytes(32)) . PHP_EOL;"

# Générer les clés JWT
mkdir -p config/jwt
php bin/console lexik:jwt:generate-keypair --skip-if-exists
```

## 4. Configuration des permissions

```bash
# Définir le propriétaire
sudo chown -R www-data:www-data /var/www/facture

# Définir les permissions
sudo chmod -R 755 /var/www/facture
sudo chmod -R 775 /var/www/facture/var
sudo chmod -R 775 /var/www/facture/public/uploads
```

## 5. Configuration Nginx

Créer le fichier de configuration :

```bash
sudo nano /etc/nginx/sites-available/facture
```

Contenu de la configuration :

```nginx
server {
    listen 80;
    server_name votre-domaine.com www.votre-domaine.com;
    root /var/www/facture/public;

    index index.php index.html;

    # Logs
    access_log /var/log/nginx/facture_access.log;
    error_log /var/log/nginx/facture_error.log;

    # Sécurité
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;

    # Taille maximale des uploads
    client_max_body_size 50M;

    # Bloquer l'accès aux fichiers sensibles
    location ~ /\. {
        deny all;
        access_log off;
        log_not_found off;
    }

    # Optimisation des fichiers statiques
    location ~* \.(jpg|jpeg|png|gif|ico|css|js|svg|woff|woff2|ttf|eot)$ {
        expires 1y;
        add_header Cache-Control "public, immutable";
        access_log off;
    }

    # Gestion des requêtes PHP
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
        fastcgi_read_timeout 300;
        fastcgi_buffer_size 128k;
        fastcgi_buffers 4 256k;
        fastcgi_busy_buffers_size 256k;
    }

    # Compression
    gzip on;
    gzip_vary on;
    gzip_min_length 1024;
    gzip_types text/plain text/css text/xml text/javascript application/x-javascript application/xml+rss application/json application/javascript;
}
```

Activer le site :

```bash
# Créer le lien symbolique
sudo ln -s /etc/nginx/sites-available/facture /etc/nginx/sites-enabled/

# Supprimer la configuration par défaut (optionnel)
sudo rm /etc/nginx/sites-enabled/default

# Tester la configuration
sudo nginx -t

# Redémarrer Nginx
sudo systemctl restart nginx
```

## 6. Configuration PHP-FPM

Optimiser PHP-FPM pour la production :

```bash
sudo nano /etc/php/8.2/fpm/php.ini
```

Modifications recommandées :

```ini
memory_limit = 256M
upload_max_filesize = 50M
post_max_size = 50M
max_execution_time = 300
date.timezone = Europe/Paris
```

Optimiser OPcache :

```bash
sudo nano /etc/php/8.2/fpm/conf.d/10-opcache.ini
```

```ini
opcache.enable=1
opcache.memory_consumption=256
opcache.interned_strings_buffer=16
opcache.max_accelerated_files=20000
opcache.validate_timestamps=0
opcache.save_comments=1
opcache.fast_shutdown=1
```

Redémarrer PHP-FPM :

```bash
sudo systemctl restart php8.2-fpm
```

## 7. Installation de l'application

```bash
cd /var/www/facture

# Créer la base de données (si elle n'existe pas déjà)
php bin/console doctrine:database:create --if-not-exists

# Exécuter les migrations
php bin/console doctrine:migrations:migrate --no-interaction

# Vider le cache
php bin/console cache:clear --env=prod --no-debug

# Réchauffer le cache
php bin/console cache:warmup --env=prod

# Créer un utilisateur administrateur
php bin/console app:create-admin
```

## 8. Configuration SSL avec Let's Encrypt (Recommandé)

```bash
# Installer Certbot
sudo apt install certbot python3-certbot-nginx -y

# Obtenir un certificat SSL
sudo certbot --nginx -d votre-domaine.com -d www.votre-domaine.com

# Le certificat sera renouvelé automatiquement
```

## 9. Configuration du firewall

```bash
# Autoriser HTTP et HTTPS
sudo ufw allow 80/tcp
sudo ufw allow 443/tcp

# Autoriser SSH (si nécessaire)
sudo ufw allow 22/tcp

# Activer le firewall
sudo ufw enable
```

## 10. Configuration des services au démarrage

```bash
# Activer les services au démarrage
sudo systemctl enable nginx
sudo systemctl enable php8.2-fpm
sudo systemctl enable mysql
```

## 11. Script de sauvegarde

Créer le script de sauvegarde :

```bash
sudo nano /usr/local/bin/backup-facture.sh
```

```bash
#!/bin/bash

BACKUP_DIR="/var/backups/facture"
DATE=$(date +%Y%m%d_%H%M%S)
DB_NAME="saasinfo"
DB_USER="saasinfo"
DB_PASS="votre-mot-de-passe"

mkdir -p $BACKUP_DIR

# Sauvegarder la base de données
mysqldump -u$DB_USER -p$DB_PASS $DB_NAME > $BACKUP_DIR/backup_$DATE.sql
gzip $BACKUP_DIR/backup_$DATE.sql

# Sauvegarder les uploads
tar -czf $BACKUP_DIR/uploads_$DATE.tar.gz -C /var/www/facture/public uploads

# Supprimer les sauvegardes de plus de 30 jours
find $BACKUP_DIR -name "backup_*.sql.gz" -mtime +30 -delete
find $BACKUP_DIR -name "uploads_*.tar.gz" -mtime +30 -delete

echo "Sauvegarde terminée: backup_$DATE.sql.gz et uploads_$DATE.tar.gz"
```

Rendre exécutable :

```bash
sudo chmod +x /usr/local/bin/backup-facture.sh
```

Ajouter au cron pour une sauvegarde quotidienne :

```bash
sudo crontab -e
# Ajouter cette ligne pour une sauvegarde quotidienne à 2h du matin
0 2 * * * /usr/local/bin/backup-facture.sh >> /var/log/facture-backup.log 2>&1
```

## 12. Script de déploiement

Créer un script de déploiement :

```bash
nano /var/www/facture/deploy-prod.sh
```

```bash
#!/bin/bash

set -e

echo "🚀 Début du déploiement..."

cd /var/www/facture

# Récupérer les dernières modifications
echo "📥 Récupération des modifications..."
git pull origin main

# Installer les dépendances
echo "📦 Installation des dépendances..."
composer install --no-dev --optimize-autoloader --no-interaction

# Exécuter les migrations
echo "🔄 Exécution des migrations..."
php bin/console doctrine:migrations:migrate --no-interaction

# Vider le cache
echo "🧹 Vidage du cache..."
php bin/console cache:clear --env=prod --no-debug

# Réchauffer le cache
echo "🔥 Réchauffement du cache..."
php bin/console cache:warmup --env=prod

# Définir les permissions
echo "🔐 Configuration des permissions..."
sudo chown -R www-data:www-data /var/www/facture
sudo chmod -R 755 /var/www/facture
sudo chmod -R 775 /var/www/facture/var
sudo chmod -R 775 /var/www/facture/public/uploads

echo "✅ Déploiement terminé avec succès!"
```

Rendre exécutable :

```bash
chmod +x /var/www/facture/deploy-prod.sh
```

## 13. Mise à jour de l'application

Pour mettre à jour l'application :

```bash
cd /var/www/facture
./deploy-prod.sh
```

## 14. Surveillance et logs

### Voir les logs

```bash
# Logs Nginx
sudo tail -f /var/log/nginx/facture_error.log
sudo tail -f /var/log/nginx/facture_access.log

# Logs PHP-FPM
sudo tail -f /var/log/php8.2-fpm.log

# Logs Symfony
tail -f /var/www/facture/var/log/prod.log

# Logs MySQL
sudo tail -f /var/log/mysql/error.log
```

### Vérifier l'état des services

```bash
sudo systemctl status nginx
sudo systemctl status php8.2-fpm
sudo systemctl status mysql
```

## 15. Dépannage

### Le site ne charge pas

```bash
# Vérifier Nginx
sudo nginx -t
sudo systemctl status nginx

# Vérifier PHP-FPM
sudo systemctl status php8.2-fpm

# Vérifier les permissions
ls -la /var/www/facture
```

### Erreurs de permissions

```bash
sudo chown -R www-data:www-data /var/www/facture
sudo chmod -R 755 /var/www/facture
sudo chmod -R 775 /var/www/facture/var
sudo chmod -R 775 /var/www/facture/public/uploads
```

### Problèmes de base de données

```bash
# Tester la connexion
mysql -u saasinfo -p saasinfo

# Vérifier les logs MySQL
sudo tail -f /var/log/mysql/error.log
```

## 16. Sécurité

- ✅ Changez tous les mots de passe par défaut
- ✅ Utilisez des mots de passe forts
- ✅ Activez le firewall (UFW)
- ✅ Configurez SSL/HTTPS
- ✅ Limitez l'accès SSH
- ✅ Configurez des sauvegardes automatiques
- ✅ Surveillez les logs régulièrement
- ✅ Gardez le système à jour : `sudo apt update && sudo apt upgrade`

## 17. Configuration Apache (Alternative à Nginx)

Si vous préférez Apache :

```bash
sudo apt install apache2 libapache2-mod-php8.2 -y
```

Créer le fichier de configuration :

```bash
sudo nano /etc/apache2/sites-available/facture.conf
```

```apache
<VirtualHost *:80>
    ServerName votre-domaine.com
    ServerAlias www.votre-domaine.com
    DocumentRoot /var/www/facture/public

    <Directory /var/www/facture/public>
        AllowOverride All
        Require all granted
    </Directory>

    ErrorLog ${APACHE_LOG_DIR}/facture_error.log
    CustomLog ${APACHE_LOG_DIR}/facture_access.log combined
</VirtualHost>
```

Activer le site et les modules nécessaires :

```bash
sudo a2ensite facture.conf
sudo a2enmod rewrite
sudo a2enmod ssl
sudo systemctl restart apache2
```

## Résumé des commandes importantes

```bash
# Déploiement
cd /var/www/facture && ./deploy-prod.sh

# Sauvegarde
sudo /usr/local/bin/backup-facture.sh

# Logs
sudo tail -f /var/log/nginx/facture_error.log

# Redémarrer les services
sudo systemctl restart nginx
sudo systemctl restart php8.2-fpm

# Mettre à jour
cd /var/www/facture && git pull && ./deploy-prod.sh
```

Bon déploiement ! 🚀

