<?php

namespace App\Repository;

use App\Entity\Produit;
use App\Entity\Magasin;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

class ProduitRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Produit::class);
    }

    public function findByMagasin(
        Magasin $magasin,
        ?string $search = null,
        ?int $categorieId = null,
        ?int $sousCategorieId = null,
        ?string $codeBarres = null,
        ?string $referenceInterne = null,
        ?string $statut = null,
        ?int $fournisseurId = null,
        ?float $prixMin = null,
        ?float $prixMax = null,
        ?string $imei = null,
        ?string $numeroSerie = null,
        ?array $caracteristiques = null,
        ?string $orderBy = 'date_entree_desc'
    ): array {
        $qb = $this->createQueryBuilder('p')
            ->leftJoin('p.sousCategorie', 'sc')
            ->leftJoin('p.fournisseur', 'f')
            ->where('p.magasin = :magasin')
            ->andWhere('p.actif = true')
            ->setParameter('magasin', $magasin);

        if ($search) {
            $qb->andWhere('p.nom LIKE :search OR p.description LIKE :search OR p.referenceInterne LIKE :search')
                ->setParameter('search', '%' . $search . '%');
        }

        if ($categorieId) {
            $qb->andWhere('p.categorie = :categorieId')
                ->setParameter('categorieId', $categorieId);
        }

        if ($sousCategorieId) {
            $qb->andWhere('p.sousCategorie = :sousCategorieId')
                ->setParameter('sousCategorieId', $sousCategorieId);
        }

        if ($codeBarres) {
            $qb->andWhere('p.codeBarres = :codeBarres')
                ->setParameter('codeBarres', $codeBarres);
        }

        if ($referenceInterne) {
            $qb->andWhere('p.referenceInterne LIKE :referenceInterne')
                ->setParameter('referenceInterne', '%' . $referenceInterne . '%');
        }

        if ($statut) {
            $qb->andWhere('p.statut = :statut')
                ->setParameter('statut', $statut);
        }

        if ($fournisseurId) {
            $qb->andWhere('p.fournisseur = :fournisseurId')
                ->setParameter('fournisseurId', $fournisseurId);
        }

        if ($prixMin !== null) {
            $qb->andWhere('p.prixAchat >= :prixMin')
                ->setParameter('prixMin', $prixMin);
        }

        if ($prixMax !== null) {
            $qb->andWhere('p.prixAchat <= :prixMax')
                ->setParameter('prixMax', $prixMax);
        }

        if ($imei) {
            $qb->andWhere('p.imei LIKE :imei')
                ->setParameter('imei', '%' . $imei . '%');
        }

        if ($numeroSerie) {
            $qb->andWhere('p.numeroSerie LIKE :numeroSerie')
                ->setParameter('numeroSerie', '%' . $numeroSerie . '%');
        }

        // Filtre par caractéristiques (recherche dans le JSON)
        // Note: On filtre après récupération car JSON_EXTRACT peut être complexe selon la version MySQL
        // On va filtrer en PHP après la requête pour plus de compatibilité

        // Appliquer le tri
        switch ($orderBy) {
            case 'nom_asc':
                $qb->orderBy('p.nom', 'ASC');
                break;
            case 'nom_desc':
                $qb->orderBy('p.nom', 'DESC');
                break;
            case 'prix_achat_asc':
                $qb->orderBy('p.prixAchat', 'ASC');
                break;
            case 'prix_achat_desc':
                $qb->orderBy('p.prixAchat', 'DESC');
                break;
            case 'prix_vente_asc':
                $qb->orderBy('p.prixVenteRecommande', 'ASC');
                break;
            case 'prix_vente_desc':
                $qb->orderBy('p.prixVenteRecommande', 'DESC');
                break;
            case 'statut_asc':
                $qb->orderBy('p.statut', 'ASC');
                break;
            case 'statut_desc':
                $qb->orderBy('p.statut', 'DESC');
                break;
            case 'date_entree_asc':
                $qb->orderBy('p.dateEntree', 'ASC');
                break;
            case 'date_entree_desc':
            default:
                $qb->orderBy('p.dateEntree', 'DESC');
                break;
        }

        $result = $qb->getQuery()->getResult();

        // Filtrer par caractéristiques en PHP (plus compatible)
        if ($caracteristiques && !empty($caracteristiques)) {
            $result = array_filter($result, function($produit) use ($caracteristiques) {
                $caracteristiquesValeurs = $produit->getCaracteristiquesValeurs() ?? [];
                foreach ($caracteristiques as $caracId => $valeur) {
                    if ($valeur) {
                        $produitValeur = $caracteristiquesValeurs[$caracId] ?? null;
                        if ($produitValeur === null) {
                            return false;
                        }
                        // Comparaison flexible (contient la valeur recherchée)
                        if (is_array($produitValeur)) {
                            if (!in_array($valeur, $produitValeur)) {
                                return false;
                            }
                        } else {
                            if (stripos((string)$produitValeur, (string)$valeur) === false) {
                                return false;
                            }
                        }
                    }
                }
                return true;
            });
        }

        return array_values($result);
    }

    /**
     * Regroupe les produits par nom et sous-catégorie pour calculer le stock total
     */
    public function getProduitsGroupes(Magasin $magasin): array
    {
        $produits = $this->findBy(['magasin' => $magasin, 'actif' => true]);
        
        $groupes = [];
        
        foreach ($produits as $produit) {
            // Clé de regroupement : nom + sous-catégorie
            $sousCategorieId = $produit->getSousCategorie() ? $produit->getSousCategorie()->getId() : 0;
            $cle = $produit->getNom() . '_' . $sousCategorieId;
            
            if (!isset($groupes[$cle])) {
                $groupes[$cle] = [
                    'nom' => $produit->getNom(),
                    'sousCategorie' => $produit->getSousCategorie(),
                    'categorie' => $produit->getCategorie(),
                    'quantiteTotal' => 0,
                    'valeurTotal' => 0.0,
                    'prixAchatMoyen' => 0.0,
                    'prixVenteMoyen' => 0.0,
                    'produits' => [],
                    'statuts' => [],
                ];
            }
            
            $groupes[$cle]['quantiteTotal'] += $produit->getQuantite();
            $groupes[$cle]['valeurTotal'] += $produit->getQuantite() * (float)$produit->getPrixAchat();
            $groupes[$cle]['produits'][] = $produit;
            
            // Compter les statuts
            $statut = $produit->getStatut();
            if (!isset($groupes[$cle]['statuts'][$statut])) {
                $groupes[$cle]['statuts'][$statut] = 0;
            }
            $groupes[$cle]['statuts'][$statut] += $produit->getQuantite();
        }
        
        // Calculer les prix moyens
        foreach ($groupes as &$groupe) {
            $totalQuantite = $groupe['quantiteTotal'];
            if ($totalQuantite > 0) {
                $groupe['prixAchatMoyen'] = $groupe['valeurTotal'] / $totalQuantite;
                
                $prixVenteTotal = 0;
                $quantiteAvecPrixVente = 0;
                foreach ($groupe['produits'] as $produit) {
                    if ($produit->getPrixVente()) {
                        $prixVenteTotal += $produit->getQuantite() * (float)$produit->getPrixVente();
                        $quantiteAvecPrixVente += $produit->getQuantite();
                    }
                }
                if ($quantiteAvecPrixVente > 0) {
                    $groupe['prixVenteMoyen'] = $prixVenteTotal / $quantiteAvecPrixVente;
                }
            }
        }
        
        // Trier par quantité décroissante
        uasort($groupes, fn($a, $b) => $b['quantiteTotal'] <=> $a['quantiteTotal']);
        
        return array_values($groupes);
    }
}

