<?php

namespace App\Form;

use App\Entity\Paiement;
use App\Entity\Achat;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\DateType;
use Symfony\Component\Form\Extension\Core\Type\MoneyType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

class PaiementType extends AbstractType
{
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $achat = $options['achat'] ?? null;

        $builder
            ->add('montant', MoneyType::class, [
                'label' => 'Montant',
                'currency' => 'MAD',
                'attr' => [
                    'class' => 'form-control'
                ]
            ])
            ->add('datePaiement', DateType::class, [
                'label' => 'Date de paiement',
                'widget' => 'single_text',
                'attr' => [
                    'class' => 'form-control'
                ]
            ])
            ->add('mode', ChoiceType::class, [
                'label' => 'Mode de paiement',
                'choices' => [
                    'Espèces' => Paiement::MODE_ESPECES,
                    'Virement' => Paiement::MODE_VIREMENT,
                    'Carte' => Paiement::MODE_CARTE,
                    'Chèque' => Paiement::MODE_CHEQUE,
                    'Autre' => Paiement::MODE_AUTRE,
                ],
                'attr' => [
                    'class' => 'form-select'
                ]
            ])
            ->add('reference', TextType::class, [
                'label' => 'Référence (numéro de transaction, chèque, etc.)',
                'required' => false,
                'attr' => [
                    'class' => 'form-control',
                    'placeholder' => 'Ex: CHQ-12345, VIR-2025-001...'
                ]
            ])
            ->add('notes', TextareaType::class, [
                'label' => 'Notes',
                'required' => false,
                'attr' => [
                    'class' => 'form-control',
                    'rows' => 3
                ]
            ]);

        if ($achat) {
            $builder->add('achat', EntityType::class, [
                'class' => Achat::class,
                'choice_label' => function (Achat $achat) {
                    return sprintf(
                        '%s - %s MAD (Reste: %s MAD)',
                        $achat->getNumeroFacture(),
                        number_format((float)$achat->getMontantTotal(), 2, ',', ' '),
                        number_format((float)$achat->getMontantRestant(), 2, ',', ' ')
                    );
                },
                'attr' => [
                    'class' => 'form-select'
                ],
                'data' => $achat
            ]);
        }
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => Paiement::class,
            'achat' => null,
        ]);
    }
}

