<?php

namespace App\Controller;

use App\Entity\Client;
use App\Form\ClientType;
use App\Repository\ClientRepository;
use App\Service\LogService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/clients', name: 'clients_')]
#[IsGranted('ROLE_USER')]
class ClientController extends AbstractController
{
    #[Route('', name: 'index', methods: ['GET'])]
    public function index(ClientRepository $clientRepository, Request $request): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $search = $request->query->get('search', '');
        $avecCredit = $request->query->get('avec_credit') === '1';

        $clients = $clientRepository->findByMagasin($magasin, $search, $avecCredit);

        return $this->render('client/index.html.twig', [
            'clients' => $clients,
            'search' => $search,
            'avecCredit' => $avecCredit,
        ]);
    }

    #[Route('/new', name: 'new', methods: ['GET', 'POST'])]
    public function new(
        Request $request,
        EntityManagerInterface $em,
        LogService $logService
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $client = new Client();
        $client->setMagasin($magasin);

        // Si c'est une requête AJAX avec FormData (depuis la modal de vente)
        if (($request->isXmlHttpRequest() || $request->headers->get('X-Requested-With') === 'XMLHttpRequest') && $request->isMethod('POST')) {
            try {
                // Récupérer les données du formulaire
                $requestData = $request->request->all();
                $data = $requestData['client'] ?? [];
                
                // Debug: logger les données reçues
                error_log('Données reçues pour création client: ' . json_encode($data));
                
                // Valider les champs requis
                if (empty($data['nom']) || empty($data['telephone'])) {
                    return new JsonResponse([
                        'success' => false,
                        'error' => 'Le nom et le téléphone sont obligatoires'
                    ], 400);
                }

                $client->setNom(trim($data['nom']));
                $client->setTelephone(trim($data['telephone']));
                $client->setCin(!empty($data['cin']) ? trim($data['cin']) : null);
                $client->setAdresse(!empty($data['adresse']) ? trim($data['adresse']) : null);
                $client->setEmail(!empty($data['email']) ? trim($data['email']) : null);

                // Gérer l'upload de la photo CIN
                $filesData = $request->files->all();
                $files = $filesData['client'] ?? [];
                $photoCinFile = $files['photoCin'] ?? null;
                if ($photoCinFile && $photoCinFile->isValid()) {
                    $uploadDir = $this->getParameter('kernel.project_dir') . '/public/uploads/clients/cin/';
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }

                    $filename = uniqid() . '.' . $photoCinFile->guessExtension();
                    $photoCinFile->move($uploadDir, $filename);
                    $client->setPhotoCin('/uploads/clients/cin/' . $filename);
                }

                $em->persist($client);
                $em->flush();

                try {
                    $logService->log($user, $magasin, 'Création client', 'Client', $client->getId(), "Client: {$client->getNom()}");
                } catch (\Exception $logError) {
                    // Ignorer les erreurs de log, ce n'est pas critique
                }

                return new JsonResponse([
                    'success' => true,
                    'client' => [
                        'id' => $client->getId(),
                        'nom' => $client->getNom(),
                        'telephone' => $client->getTelephone(),
                    ]
                ]);
            } catch (\Exception $e) {
                // Logger l'erreur pour le débogage
                error_log('Erreur création client: ' . $e->getMessage() . ' - ' . $e->getTraceAsString());
                
                return new JsonResponse([
                    'success' => false,
                    'error' => 'Erreur lors de la création: ' . $e->getMessage()
                ], 500);
            }
        }

        $form = $this->createForm(ClientType::class, $client);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Gérer l'upload de la photo CIN
            $photoCinFile = $form->get('photoCin')->getData();
            if ($photoCinFile) {
                $uploadDir = $this->getParameter('kernel.project_dir') . '/public/uploads/clients/cin/';
                if (!is_dir($uploadDir)) {
                    mkdir($uploadDir, 0755, true);
                }

                $filename = uniqid() . '.' . $photoCinFile->guessExtension();
                $photoCinFile->move($uploadDir, $filename);
                $client->setPhotoCin('/uploads/clients/cin/' . $filename);
            }

            $em->persist($client);
            $em->flush();

            $logService->log($user, $magasin, 'Création client', 'Client', $client->getId(), "Client: {$client->getNom()}");

            $this->addFlash('success', 'Client créé avec succès');

            return $this->redirectToRoute('clients_show', ['id' => $client->getId()]);
        }

        return $this->render('client/new.html.twig', [
            'client' => $client,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'show', methods: ['GET'])]
    public function show(Client $client): Response
    {
        $user = $this->getUser();
        if ($client->getMagasin() !== $user->getMagasin()) {
            throw $this->createAccessDeniedException();
        }

        // Calculer les statistiques
        $totalDepense = 0;
        $totalPaye = 0;
        $totalRestant = 0;
        $nombreAchats = 0;
        $dernierAchat = null;

        foreach ($client->getVentes() as $vente) {
            if (!$vente->isRetour()) {
                $totalDepense += (float)$vente->getMontantFinal();
                $totalPaye += (float)$vente->getMontantPaye();
                $totalRestant += (float)$vente->getMontantRestant();
                $nombreAchats++;
                
                if (!$dernierAchat || $vente->getDateVente() > $dernierAchat->getDateVente()) {
                    $dernierAchat = $vente;
                }
            }
        }

        return $this->render('client/show.html.twig', [
            'client' => $client,
            'stats' => [
                'totalDepense' => $totalDepense,
                'totalPaye' => $totalPaye,
                'totalRestant' => $totalRestant,
                'nombreAchats' => $nombreAchats,
                'dernierAchat' => $dernierAchat,
            ],
        ]);
    }

    #[Route('/{id}/edit', name: 'edit', methods: ['GET', 'POST'])]
    #[IsGranted('ROLE_MANAGER')]
    public function edit(
        Request $request,
        Client $client,
        EntityManagerInterface $em,
        LogService $logService
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($client->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(ClientType::class, $client);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Gérer l'upload de la photo CIN
            $photoCinFile = $form->get('photoCin')->getData();
            if ($photoCinFile) {
                $uploadDir = $this->getParameter('kernel.project_dir') . '/public/uploads/clients/cin/';
                if (!is_dir($uploadDir)) {
                    mkdir($uploadDir, 0755, true);
                }

                // Supprimer l'ancienne photo si elle existe
                if ($client->getPhotoCin()) {
                    $oldPath = $this->getParameter('kernel.project_dir') . '/public' . $client->getPhotoCin();
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                    }
                }

                $filename = uniqid() . '.' . $photoCinFile->guessExtension();
                $photoCinFile->move($uploadDir, $filename);
                $client->setPhotoCin('/uploads/clients/cin/' . $filename);
            }

            $em->flush();

            $logService->log($user, $magasin, 'Modification client', 'Client', $client->getId(), "Client: {$client->getNom()}");

            $this->addFlash('success', 'Client modifié avec succès');

            return $this->redirectToRoute('clients_show', ['id' => $client->getId()]);
        }

        return $this->render('client/edit.html.twig', [
            'client' => $client,
            'form' => $form,
        ]);
    }
}

